#!/usr/bin/env python3

from pprint import pprint
import ssl
import socket
import sys
import os
import yaml
from datetime import datetime, timezone
from urllib.parse import urlparse

# check /etc/sensu/agent.yml if you find ssl-warning-threshold and ssl-critical-threshold set the variables, else default to 14 and 7
ssl_warning_threshold = 14
ssl_critical_threshold = 7

with open('/etc/sensu/agent.yml', 'r') as file:
    config = yaml.load(file, Loader=yaml.SafeLoader)
    if 'ssl-warning-threshold' in config:
        ssl_warning_threshold = config['ssl-warning-threshold']
    if 'ssl-critical-threshold' in config:
        ssl_critical_threshold = config['ssl-critical-threshold']

def check_ssl_certificate(url):
    """Check SSL certificate validity for a given HTTPS URL.
    
    Returns:
        tuple: (exit_code, message) where exit_code is:
            0 = OK (>=ssl_warning_threshold days remaining)
            1 = WARNING (ssl_critical_threshold-ssl_warning_threshold days remaining)
            2 = CRITICAL (<ssl_critical_threshold days remaining)
            3 = ERROR (connection/SSL error)
    """
    try:
        # Parse the URL
        parsed = urlparse(url)
        hostname = parsed.hostname
        port = parsed.port or 443
        
        if not hostname:
            return (3, f"ERROR: Invalid URL: {url}")
        
        # Create SSL context
        context = ssl.create_default_context()
        
        # Connect to the server and get certificate
        with socket.create_connection((hostname, port), timeout=10) as sock:
            with context.wrap_socket(sock, server_hostname=hostname) as ssock:
                cert = ssock.getpeercert()
        
        # Get certificate expiration date
        expire_date_str = cert.get('notAfter')
        if not expire_date_str:
            return (3, "ERROR: Could not retrieve certificate expiration date")
        
        # Parse expiration date (format: 'Dec 31 23:59:59 2023 GMT')
        expire_date = datetime.strptime(expire_date_str, '%b %d %H:%M:%S %Y %Z')
        # Make expire_date timezone-aware (GMT is UTC)
        expire_date = expire_date.replace(tzinfo=timezone.utc)
        current_date = datetime.now(timezone.utc)
        
        # Calculate days until expiration
        days_remaining = (expire_date - current_date).days
        
        # Build result message
        message = f"- {hostname} expires in {days_remaining} days"
        
        # Determine status
        if days_remaining < ssl_critical_threshold:
            return (2, message)
        elif days_remaining < ssl_warning_threshold:
            return (1, message)
        else:
            return (0, message)
            
    except socket.gaierror as e:
        return (3, f"ERROR: Could not resolve hostname {url}: {e}")
    except socket.timeout:
        return (3, f"ERROR: Connection timeout for {url}")
    except ssl.SSLError as e:
        return (3, f"ERROR: SSL error for {url}: {e}")
    except Exception as e:
        return (3, f"ERROR: {url}: {e}")


def check_urls(urls):

    # if urls is not a list, make it a list
    if not isinstance(urls, list):
        urls = [urls]  
    
    worst_status = 0
    messages = []

    for url in urls:
        if not url.startswith('https://'):
            url = 'https://' + url

        exit_code, message = check_ssl_certificate(url)

        if exit_code != 0:
            messages.append(message)
    
        # Track worst status (higher number = worse)
    
        worst_status = max(worst_status, exit_code)


    if worst_status == 0:
        count = len(urls)
        print(f"OK: {count} hosts checked and valid within threshold ({ssl_warning_threshold} days)")
        sys.exit(0)

    if worst_status > 2:
        print(f"EXCEPTION: Check failed to run correctly")
        print("\n".join(messages))
        sys.exit(3)


    status = "CRITICAL"
    if worst_status == 1:
        status = "WARNING"
    
    count = len(messages)
    # make a better
    print(f"{status}: Certificate nearing expiration for {count} host{'s' if count != 1 else ''}")
    print("\n".join(messages))
    sys.exit(worst_status)


if __name__ == "__main__":
    
    # Case 1: Argument is given, used as url to check
    if len(sys.argv) > 1:
        url = sys.argv[1]
        check_urls([url])

    # Case 2: Check sensu config file for urls to check
    config_file = "/etc/sensu/agent.yml"
    if os.path.exists(config_file):
        with open(config_file, 'r') as file:
            config = yaml.load(file, Loader=yaml.SafeLoader)
        if 'check-http-ssl-hosts' in config:
            check_urls(config['check-http-ssl-hosts'])

    # Case 3: Check if wisol config file exists
    config_file = "/opt/wisol-install/sites.yml"
    if os.path.exists(config_file):
        with open(config_file, 'r') as file:
            config = yaml.load(file, Loader=yaml.SafeLoader)

        if 'sites' in config:
            urls_to_check = []
            for site in config['sites']:
                urls_to_check.append(site)
            check_urls(urls_to_check)

    # Case 4: If nothing is specified, default to checking the hostname
    hostname = socket.gethostname()
    check_urls([hostname]) 
