#! /usr/bin/env python

import os
import sys
import time
import subprocess
import re

def run_command(cmd):
    """Run a shell command and return the output"""
    try:
        result = subprocess.check_output(cmd, shell=True, stderr=subprocess.STDOUT)
        return result.strip()
    except subprocess.CalledProcessError as e:
        print("Error running command '%s': %s" % (cmd, e))
        sys.exit(3)
        
def get_host_uuid():
    """Get the host UUID from xensource-inventory"""
    try:
        with open('/etc/xensource-inventory', 'r') as f:
            content = f.read()
            match = re.search(r"INSTALLATION_UUID='([a-z0-9\-]*)'", content)
            if match:
                return match.group(1)
    except FileNotFoundError:
        print("ERROR - Cannot find /etc/xensource-inventory")
        sys.exit(3)
    except Exception as e:
        print("ERROR - Failed to read host UUID: %s" % e)
        sys.exit(3)
    return None

def get_master_uuid():
    """Get the master UUID from pool-list"""
    output = run_command("sudo /opt/xensource/bin/xe pool-list")
    for line in output.split('\n'):
        if 'master' in line:
            parts = line.split()
            if len(parts) >= 4:
                return parts[3]
    return None

def get_sr_uuids():
    """Get all SR UUIDs for the specified types"""
    sr_types = ['nfs', 'ext', 'lvmoiscsi']
    sr_uuids = []
    
    for sr_type in sr_types:
        output = run_command("%s sr-list type=%s" % (XE, sr_type))
        for line in output.split('\n'):
            if 'uuid' in line:
                parts = line.split()
                if len(parts) >= 5:
                    sr_uuids.append(parts[4])
    
    return sr_uuids

def check_sr_usage(sr_uuid):
    """Check usage for a specific SR and return warning/critical status"""


    try:
        # Get physical utilisation
        pu_output = run_command("%s sr-list uuid=%s params=physical-utilisation" % (XE, sr_uuid))
        pu = float(pu_output.split()[-1])
    
        
        # Get physical size
        ps_output = run_command("%s sr-list uuid=%s params=physical-size" % (XE, sr_uuid))
        ps = float(ps_output.split()[-1])
        
        # Calculate percentage
        sr_pc = (pu / ps) * 100
        
        # Get SR name
        name_output = run_command("%s sr-list uuid=%s params=name-label" % (XE, sr_uuid))
        sr_name = name_output[22:] if len(name_output) > 22 else name_output

        return sr_name, sr_pc
        
    except (ValueError, IndexError) as e:
        print("Warning: Could not parse SR %s data: %s" % (sr_uuid,e))
        return None, 0

def main():
    if len(sys.argv) != 3:
        print("Usage: check-xen-storage.py <warning_threshold> <critical_threshold>")
        print("Example: check-xen-storage.py 80 90")
        sys.exit(3)
    
    try:
        WARN = int(sys.argv[1])
        CRIT = int(sys.argv[2])
    except ValueError:
        print("ERROR - Warning and critical thresholds must be integers")
        sys.exit(3)
    
    global XE
    XE = "sudo /opt/xensource/bin/xe"
    
    # Get host UUID
    host_uuid = get_host_uuid()
    if not host_uuid:
        print("ERROR - Could not determine host UUID")
        sys.exit(3)

    # Get master UUID
    master_uuid = get_master_uuid()
    if not master_uuid:
        print("ERROR - Could not determine master UUID")
        sys.exit(3)

    # Check if this host is the master
    if host_uuid != master_uuid:
        print("OK - Host is slave - storage only checked on master host")
        sys.exit(0)
    
    # Get all SR UUIDs to check
    sr_uuids = get_sr_uuids()

    warning_messages = []
    critical_messages = []
    
    # Check each SR
    for sr_uuid in sr_uuids:
        sr_name, sr_pc = check_sr_usage(sr_uuid)
        if sr_name and sr_pc > 0:
            if sr_pc > CRIT:
                critical_messages.append("%s usage over at %.2f%%" % (sr_name, sr_pc))
            elif sr_pc > WARN:
                warning_messages.append("%s usage over at %.2f%%" % (sr_name, sr_pc))
    
    # Output results
    if critical_messages:
        print("CRITICAL - %d SR(s) in critical usage" % len(critical_messages))
        for msg in critical_messages:
            print(msg)
        if warning_messages:
            print()
            for msg in warning_messages:
                print(msg)
        sys.exit(2)
    
    if warning_messages:
        print("WARNING - %d SR(s) in warning usage" % len(warning_messages))
        for msg in warning_messages:
            print(msg)
        sys.exit(1)
    
    # Everything OK
    print("OK - All SRs within usage range")
    sys.exit(0)

if __name__ == "__main__":
    main()
